-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/contrib/gsql/query_handler.adb,v $
--  Description     : GUI for the SQL client
--  Author          : Michael Erdmann                                        --
--  Created         : 8.8.2001                                               --
--  Last Modified By: $Author: persan $
--  Last Modified On: $Date: 2007/09/06 20:17:53 $
--  Status          : $State: Exp $
--
--  Copyright (C) 2000 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  Functional description                                                   --
--  ======================                                                   --
--                                                                           --
--  Author                                                                   --
--  ======                                                                   --
--                                                                           --
--  Author: Michael Erdmann <michael.erdmann@snafu.de>                       --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  GNADE is implemented to work with GNAT, the GNU Ada compiler.            --
--                                                                           --
-------------------------------------------------------------------------------
with Ada.Text_IO;              use Ada.Text_IO;

with Text_Block;               use Text_Block;
with Contents_Manager.GUI;     use Contents_Manager.GUI;
with Datasource_Handler;       use Datasource_Handler;
with Preferences;              use Preferences;

with GUI_Preferences;          use GUI_Preferences;
with GUI_Logging;              use GUI_Logging;
with GUI_Tools;                use GUI_Tools;
with GUI_Worksheet;            use GUI_Worksheet;

package body Query_Handler  is


   -- ********************************************************************* --
   -- **                 L O C A L   D A T A                             ** --
   -- ********************************************************************* --

   Query_Section : Element_ID := Null_Element_ID;

   -- ********************************************************************* --
   -- **              L O C A L    P R O C E D U R E S                   ** --
   -- ********************************************************************* --

   -----------------
   -- Instanciate --
   -----------------
   function Instanciate (
                         Key    : in String) return Contents_Handler.Handle is
      pragma Unreferenced (Key);
      Result : constant Handle := new Object;
   begin
      return Contents_Handler.Handle (Result);
   end Instanciate;

   -- ********************************************************************* --
   -- **               P U B L I C   P R O C E D U R E S                 ** --
   -- **                     ( CLASS - Methods )                         ** --
   -- ********************************************************************* --

   ----------------
   -- Initialize --
   ----------------
   procedure Initialize is
   -- initialize the package (class)
      Item : constant Handle := new Object;
   begin
      Item.Constructor := Instanciate'Access;
      Query_Section := Add_Element (
                                    Title => "Queries",
                                    Content => Contents_Handler.Handle (Item)
                                   );
      Contents_Manager.Register (Query_Prefix, Item.Constructor);
      Contents_Manager.Initialize (Query_Prefix, Query_Section);
   end Initialize;

   --------------
   -- Finalize --
   --------------
   procedure Finalize is
   -- finalize the package by writing out all contents elements
   -- into the project file.
   begin
      Contents_Manager.Finalize (Query_Prefix);
   end Finalize;

   ----------------
   -- Initialize --
   ----------------
   procedure Initialize (
                         This : in out Object) is
   -- initialize the query (instance)
   begin
      Initialize (Contents_Handler.Object (This));
      Put_Line ("Initialize: " & Key ( This ) );

      This.Id := GUI.Add_Element ( Key (This), Query_Section, Self (This) );
   end Initialize;

   --------------
   -- Finalize --
   --------------
   procedure Finalize (
                       This : in out Object ) is
      Name : constant Unbounded_String := Key ( This );
   begin
      if GUI_Worksheet.Is_Open ( Name ) then
         GUI_Worksheet.Remove_Sheet ( Name );
      end if;
      Project_File.Remove ( Query_Prefix, Key (This) );

      GUI.Delete_Element ( This.Id );
      Finalize ( Contents_Handler.Object (This) );
   end Finalize;

   -- ********************************************************************* --
   -- **               P U B L I C   P R O C E D U R E S                 ** --
   -- **                     ( INSATNCE - Methods )                      ** --
   -- ********************************************************************* --

   -----------
   -- Value --
   -----------
   procedure Value (
                    This : in Object;
                    Text : out Text_Buffer;
                    Last : out Positive ) is
   -- this procedure retrieves the contents of the GUI and sets the
   -- object name.
   begin
      GUI_Editor.Get_Result ( This.Editor, Text, Last);
   end Value;

   ----------
   -- Save --
   ----------
   procedure Save (
                   Item    : in out Object;
                   Value   : in Text_Buffer;
                   Name    : in String := "" ) is
   -- R.1  If the object is already in the project file, then simply
   --      update the data
   -- R.2  if the object is not in the project file, assume, that the
   --      element with the specified name has to be created newly.
      Current : constant String := Key ( Item );
      NewObj  : Handle;
   begin
      Item.Name := Value (1);

      if not Section_Exists ( Query_Prefix, To_String ( Item.Name ) ) then   -- R.2
         NewObj := Handle (Contents_Manager.Create ( Query_Prefix, To_String (Item.Name)) );
         Gui_Worksheet.Remove_Sheet ( Item.Editor );
      end if;
      -- R.1, R.2
      Project_File.Update ( Query_Prefix, To_String (Item.Name), Value);
   end Save;

   ----------
   -- Load --
   ----------
   procedure Load (
                   Item        : in out Object;
                   Value       : out Text_Buffer;
                   Last        : out Natural ) is

      Description : constant Text_Buffer := (
                                             To_Unbounded_String ( "Fill in your description" ),
                                             To_Unbounded_String ( "Author: " & Preference (Prf_Author))
                                            );

      Query       : constant Text_Buffer := (
                                             To_Unbounded_String ( "SELECT * FROM <relation> "),
                                             To_Unbounded_String ( "WHERE ..... ")
                                            );
   begin
      Project_File.Get (
                        Query_Prefix,
                        Key (Item),
                        Value,
                        Last
                       );

   exception
      when Section_Not_Found =>
         Last := Value'First;
         Value (Last) := Item.Name;
         Last := Last + 1;
         Put_Text_Block ( Value, Description, Last );
         Put_Text_Block ( Value, Query, Last );
   end Load;

   --------------
   -- Selected --
   --------------
   procedure Selected (
                       This        : in out Object ) is
   --
   -- R.1 - If only the class is selected the toolbar is
   --       activated for the query class.
   --
   -- R.2 - If an instance is selected, the instance is
   --       loaded into the editor.
   --
   begin
      if This.Constructor /= null then                              -- R.1
         GUI_Tools.Activate ( Query_Prefix  );
      else                                                          -- R.2
         Open_Editor ( This );
      end if;
   end Selected;

   -----------------
   -- Open_Editor --
   -----------------
   procedure Open_Editor (
                          This : in out Object ) is
   -- load the referenced instance into the query editor
      Name : Unbounded_String := Key ( This );
   begin
      if not GUI_Worksheet.Is_Open ( Name ) then
         Initialize ( This.Editor );
         GUI_Worksheet.Add_Sheet ( This.Editor, Self (This), Name );
      else
         GUI_Worksheet.Bring_In_Front ( Name );
      end if;
   end Open_Editor;

   ------------------
   -- Close_Editor --
   ------------------
   procedure Close_Editor (
                           This : in out Object ) is
      Name : Unbounded_String := Key ( This );
   begin
      GUI_Worksheet.Remove_Sheet ( Name );
   end Close_Editor;

   -------------
   -- Execute --
   -------------
   procedure Execute (
                      This : in out Object ) is
   -- execute the current editor contents
      Name : Unbounded_String := Key ( This );
      Text : Text_Buffer ( 1 .. Max_Section_Length );
      Last : Positive;
   begin
      if Datasource_Handler.Current_Database = Not_Attached then
         Error ("No data base connected");
         return;
      end if;

      Log ("Executing " & To_String (Name) );
      GUI_Editor.Query (This.Editor, Text, Last);
      Datasource_Handler.Execute ( To_String ( Text (1 .. Last ) ) );
   end Execute;

end Query_Handler;
