(*============================================================================*)
(*        FILE: reduct.sml                                                    *)
(*                                                                            *)
(* DESCRIPTION: General "reduction" properties of binary relations, such as   *)
(*              normalization, confluence etc. Theorems about the relations   *)
(*              between them, e.g. Newman's Lemma.                            *)
(*                                                                            *)
(*      AUTHOR: John Harrison                                                 *)
(*              University of Cambridge Computer Laboratory                   *)
(*              New Museums Site                                              *)
(*              Pembroke Street                                               *)
(*              Cambridge CB2 3QG                                             *)
(*              England.                                                      *)
(*              jrh@cl.cam.ac.uk                                              *)
(*                                                                            *)
(*        DATE: 29th May 1993                                                 *)
(*                                                                            *)
(*  TRANSLATED: John Harrison, 31st May 1993                                  *)
(*============================================================================*)

new_theory "REDUCT";

load_library{lib=find_library "ind_def",theory="-"};

open Inductive_def;;

(*----------------------------------------------------------------------------*)
(* For compatibility                                                          *)
(*----------------------------------------------------------------------------*)

val num_Axiom = theorem "prim_rec" "num_Axiom";

val num_CASES = theorem "arithmetic" "num_CASES";

fun W f x = f x x;

fun GEN_REWRITE_CONV cnv l1 l2 :conv = cnv
  (FIRST_CONV(map REWR_CONV (l1@l2)));

fun GEN_REWRITE_RULE cnv l1 l2 = CONV_RULE(GEN_REWRITE_CONV cnv l1 l2);

fun GEN_REWRITE_TAC cnv l1 l2 = CONV_TAC(GEN_REWRITE_CONV cnv l1 l2);

val prove_thm = store_thm;

exception Unchanged;

fun CHANGED_TAC tac gl =
  let val (sg,just) = tac gl in
      if sg = [gl] then raise Unchanged
      else (sg,just)
  end;

(*----------------------------------------------------------------------------*)
(* Useful oddments                                                            *)
(*----------------------------------------------------------------------------*)

val LAND_CONV = RATOR_CONV o RAND_CONV;

val TAUT_CONV =
  let fun vl w t = type_of t = (==`:bool`==) andalso
  can (find_term is_var) t andalso free_in t w in
  C (curry prove)
  (REPEAT GEN_TAC THEN (REPEAT o CHANGED_TAC o W)
   (C (curry op THEN) (REWRITE_TAC[]) o BOOL_CASES_TAC o hd o
   sort free_in o
   W(find_terms o vl) o snd)) end;

fun ANTE_RES_THEN ttac th = FIRST_ASSUM(ttac o C MATCH_MP th);

val RULE_INDUCT_TAC = C W STRIP_ASSUME_TAC o RULE_INDUCT_THEN;

(*----------------------------------------------------------------------------*)
(* We use the RSTC theory a great deal                                        *)
(*----------------------------------------------------------------------------*)

new_parent "RSTC";

Add_to_sml.add_theory_to_sml "RSTC";

(*----------------------------------------------------------------------------*)
(* Useful lemmas: essentially equivalent forms of wellfoundedness             *)
(*----------------------------------------------------------------------------*)

val SEQ_EXISTS_IMP = prove_thm("SEQ_EXISTS_IMP",
  (--`!R (P:'a->bool). (?x. P x) /\ (!x. P x ==> ?y. P y /\ R x y) ==>
        ?seq. (!n. P(seq n)) /\ (!n. R (seq n) (seq (SUC n)))`--),
  REPEAT STRIP_TAC THEN
  MP_TAC(ISPECL [(--`x:'a`--), (--`\(s:'a) (n:num). @y. P y /\ (R:'a->'a->bool) s y`--)]
    num_Axiom) THEN DISCH_THEN(MP_TAC o EXISTENCE) THEN BETA_TAC THEN
  DISCH_THEN(X_CHOOSE_THEN (--`seq:num->'a`--) STRIP_ASSUME_TAC) THEN
  EXISTS_TAC (--`seq:num->'a`--) THEN
  SUBGOAL_THEN (--`!n. P(seq (SUC n)) /\ (R:'a->'a->bool) (seq n) (seq(SUC n))`--)
  ASSUME_TAC THENL
   [INDUCT_TAC THENL
     [ASM_REWRITE_TAC[], FIRST_ASSUM(fn th =>  REWRITE_TAC[SPEC (--`SUC n`--) th])] THEN
    BETA_TAC THEN CONV_TAC SELECT_CONV THEN
    FIRST_ASSUM MATCH_MP_TAC THEN ASM_REWRITE_TAC[],
    ASM_REWRITE_TAC[] THEN X_GEN_TAC (--`n:num`--) THEN
    STRUCT_CASES_TAC(SPEC (--`n:num`--) num_CASES) THEN ASM_REWRITE_TAC[]]);

val SEQ_EXISTS = prove_thm("SEQ_EXISTS",
  (--`!R. (?P:'a->bool. (?x. P x) /\ (!x. P x ==> ?y. P y /\ R x y)) =
       (?seq. !n. R (seq n) (seq (SUC n)))`--),
  REPEAT GEN_TAC THEN EQ_TAC THENL
   [DISCH_THEN(CHOOSE_THEN (X_CHOOSE_TAC (--`seq:num->'a`--) o
      MATCH_MP SEQ_EXISTS_IMP)) THEN EXISTS_TAC (--`seq:num->'a`--) THEN
    ASM_REWRITE_TAC[],
    DISCH_THEN(X_CHOOSE_TAC (--`seq:num->'a`--)) THEN
    EXISTS_TAC (--`\x:'a. ?n:num. x = seq(n)`--) THEN
    BETA_TAC THEN REWRITE_TAC[NOT_IMP] THEN CONJ_TAC THENL
     [MAP_EVERY EXISTS_TAC [(--`seq 0 :'a`--), (--`0`--)] THEN REFL_TAC,
      X_GEN_TAC (--`x:'a`--) THEN DISCH_THEN(X_CHOOSE_TAC (--`n:num`--)) THEN
      EXISTS_TAC (--`seq(SUC n):'a`--) THEN ASM_REWRITE_TAC[] THEN
      EXISTS_TAC (--`SUC n`--) THEN REFL_TAC]]);

(*----------------------------------------------------------------------------*)
(* Normality of a term w.r.t. a reduction relation                            *)
(*----------------------------------------------------------------------------*)

val NORMAL = new_definition("NORMAL",
  (--`NORMAL(R:'a->'a->bool) x = ~?y. R x y`--));

(*----------------------------------------------------------------------------*)
(* Full Church-Rosser property.                                               *)
(*                                                                            *)
(* Note that we deviate from most term rewriting literature which call this   *)
(* the "diamond property" and calls a relation "Church-Rosser" iff its RTC    *)
(* has the diamond property. But this seems simpler and more natural.         *)
(*----------------------------------------------------------------------------*)

val CR = new_definition("CR",
  (--`CR(R:'a->'a->bool) =
    !x y1 y2. R x y1 /\ R x y2 ==> ?z. R y1 z /\ R y2 z`--));

(*----------------------------------------------------------------------------*)
(* Weak Church-Rosser property, i.e. the rejoining may take several steps.    *)
(*----------------------------------------------------------------------------*)

val WCR = new_definition("WCR",
  (--`WCR(R:'a->'a->bool) = !x y1 y2. R x y1 /\ R x y2 ==>
                                 ?z. RTC R y1 z /\ RTC R y2 z`--));

(*----------------------------------------------------------------------------*)
(* (Weak) normalization: every term has a normal form.                        *)
(*----------------------------------------------------------------------------*)

val WN = new_definition("WN",
  (--`WN(R:'a->'a->bool) = !x. ?y. RTC R x y /\ NORMAL(R) y`--));

(*----------------------------------------------------------------------------*)
(* Strong normalization: every reduction sequence terminates (aka Noetherian) *)
(*----------------------------------------------------------------------------*)

val SN = new_definition("SN",
  (--`SN(R:'a->'a->bool) = ~?seq. !n. R (seq n) (seq (SUC n))`--));

(*----------------------------------------------------------------------------*)
(* Alternative "preservation" form of SN definition which is more useful.     *)
(*----------------------------------------------------------------------------*)

val SN_PRESERVE = prove_thm("SN_PRESERVE",
  (--`!R:'a->'a->bool. SN(R) = !P. (!x. P x ==> ?y. P y /\ R x y) ==> ~?x. P x`--),
  GEN_TAC THEN REWRITE_TAC[SN, TAUT_CONV (--`(a ==> ~b) = ~(b /\ a)`--)] THEN
  GEN_REWRITE_TAC I [] [TAUT_CONV (--`(~a = b) = (a = ~b)`--)] THEN
  CONV_TAC(RAND_CONV NOT_FORALL_CONV) THEN
  REWRITE_TAC[SEQ_EXISTS]);

(*----------------------------------------------------------------------------*)
(* "Noetherian induction" for strongly normalizing relation                   *)
(*----------------------------------------------------------------------------*)

val SN_NOETHERIAN = prove_thm("SN_NOETHERIAN",
  (--`!R:'a->'a->bool. SN(R) = !P. (!x. (!y. R x y ==> P y) ==> P x) ==> !x. P x`--),
  GEN_TAC THEN REWRITE_TAC[SN_PRESERVE] THEN
  CONV_TAC((LAND_CONV o RAND_CONV o ABS_CONV o LAND_CONV o ONCE_DEPTH_CONV)
    CONTRAPOS_CONV) THEN
  CONV_TAC(ONCE_DEPTH_CONV NOT_EXISTS_CONV) THEN
  REWRITE_TAC[TAUT_CONV (--`a ==> b = b \/ ~a`--), DE_MORGAN_THM] THEN
  EQ_TAC THEN DISCH_TAC THEN X_GEN_TAC (--`P:'a->bool`--) THEN
  POP_ASSUM(MP_TAC o SPEC (--`\x:'a. ~(P x)`--)) THEN BETA_TAC THEN
  REWRITE_TAC[]);

(*----------------------------------------------------------------------------*)
(* Normality is preserved by transitive closure                               *)
(*----------------------------------------------------------------------------*)

val NORMAL_TC = prove_thm("NORMAL_TC",
  (--`!R:'a->'a->bool. NORMAL(TC R) x = NORMAL(R) x`--),
  GEN_TAC THEN REWRITE_TAC[NORMAL] THEN
  AP_TERM_TAC THEN EQ_TAC THEN DISCH_THEN(X_CHOOSE_TAC (--`y:'a`--)) THENL
   [POP_ASSUM(DISJ_CASES_TAC o ONCE_REWRITE_RULE[TC_CASES_R]) THENL
     [EXISTS_TAC (--`y:'a`--),
      POP_ASSUM(X_CHOOSE_TAC (--`z:'a`--)) THEN EXISTS_TAC (--`z:'a`--)],
    EXISTS_TAC (--`y:'a`--) THEN MATCH_MP_TAC TC_INC] THEN
  ASM_REWRITE_TAC[]);

(*----------------------------------------------------------------------------*)
(* Hence so is normalization                                                  *)
(*----------------------------------------------------------------------------*)

val WN_TC = prove_thm("WN_TC",
  (--`!R:'a->'a->bool. WN(TC R) = WN R`--),
  REWRITE_TAC[WN, NORMAL_TC, RTC, TC_IDEMP]);

(*----------------------------------------------------------------------------*)
(* Strong normalization is too in fact.                                       *)
(*----------------------------------------------------------------------------*)

val SN_TC = prove_thm("SN_TC",
  (--`!R:'a->'a->bool. SN(TC R) = SN R`--),
  GEN_TAC THEN EQ_TAC THENL
   [REWRITE_TAC[SN] THEN CONV_TAC CONTRAPOS_CONV THEN REWRITE_TAC[] THEN
    DISCH_THEN(X_CHOOSE_TAC (--`seq:num->'a`--)) THEN
    EXISTS_TAC (--`seq:num->'a`--) THEN GEN_TAC THEN
    MATCH_MP_TAC TC_INC THEN ASM_REWRITE_TAC[],
    GEN_REWRITE_TAC RAND_CONV [] [SN] THEN
    CONV_TAC CONTRAPOS_CONV THEN REWRITE_TAC[] THEN
    DISCH_THEN(X_CHOOSE_TAC (--`seq:num->'a`--)) THEN
    REWRITE_TAC[SN_PRESERVE] THEN DISCH_THEN(MP_TAC o
      SPEC (--`\x:'a. ?n:num. RTC R (seq n) x /\ RTC R x (seq (SUC n))`--)) THEN
    BETA_TAC THEN REWRITE_TAC[NOT_IMP] THEN CONJ_TAC THENL
     [ALL_TAC,
      MAP_EVERY EXISTS_TAC [(--`seq(0):'a`--), (--`0`--)] THEN
      REWRITE_TAC[RTC_REFL] THEN REWRITE_TAC[RTC] THEN
      MATCH_MP_TAC RC_INC THEN ASM_REWRITE_TAC[]] THEN
    X_GEN_TAC (--`x:'a`--) THEN
    DISCH_THEN(X_CHOOSE_THEN (--`n:num`--) STRIP_ASSUME_TAC) THEN
    UNDISCH_TAC (--`RTC R x (seq(SUC n):'a)`--) THEN DISCH_THEN
    (REPEAT_TCL DISJ_CASES_THEN MP_TAC o ONCE_REWRITE_RULE[RTC_CASES_L]) THENL
     [DISCH_THEN SUBST_ALL_TAC THEN FIRST_ASSUM(MP_TAC o SPEC (--`SUC n`--)) THEN
      DISCH_THEN(DISJ_CASES_TAC o ONCE_REWRITE_RULE[TC_CASES_R]) THENL
       [EXISTS_TAC (--`seq(SUC(SUC n)):'a`--) THEN ASM_REWRITE_TAC[] THEN
        EXISTS_TAC (--`SUC n`--) THEN ASM_REWRITE_TAC[RTC_REFL] THEN
        MATCH_MP_TAC RTC_INC THEN ASM_REWRITE_TAC[],
        FIRST_ASSUM(X_CHOOSE_THEN (--`y:'a`--) STRIP_ASSUME_TAC) THEN
        EXISTS_TAC (--`y:'a`--) THEN ASM_REWRITE_TAC[] THEN
        EXISTS_TAC (--`SUC n`--) THEN CONJ_TAC THENL
         [MATCH_MP_TAC RTC_INC,
          REWRITE_TAC[RTC] THEN MATCH_MP_TAC RC_INC] THEN
        ASM_REWRITE_TAC[]],
      DISCH_THEN(X_CHOOSE_THEN (--`y:'a`--) STRIP_ASSUME_TAC) THEN
      UNDISCH_TAC (--`RTC(R:'a->'a->bool) x y`--) THEN
      DISCH_THEN(DISJ_CASES_TAC o ONCE_REWRITE_RULE[RTC_CASES_R]) THENL
       [EXISTS_TAC (--`seq(SUC n):'a`--) THEN ASM_REWRITE_TAC[] THEN
        EXISTS_TAC (--`n:num`--) THEN REWRITE_TAC[RTC_REFL] THEN
        ASM_REWRITE_TAC[RTC, RC_CASES],
        FIRST_ASSUM(X_CHOOSE_THEN (--`z:'a`--) STRIP_ASSUME_TAC) THEN
        EXISTS_TAC (--`z:'a`--) THEN ASM_REWRITE_TAC[] THEN
        EXISTS_TAC (--`n:num`--) THEN CONJ_TAC THENL
         [MATCH_MP_TAC RTC_TRANS_L THEN EXISTS_TAC (--`x:'a`--),
          MATCH_MP_TAC RTC_TRANS_L THEN EXISTS_TAC (--`y:'a`--)] THEN
        ASM_REWRITE_TAC[]]]]);

(*----------------------------------------------------------------------------*)
(* Strong normalization implies normalization                                 *)
(*----------------------------------------------------------------------------*)

val SN_WN = prove_thm("SN_WN",
  (--`!R:'a->'a->bool. SN(R) ==> WN(R)`--),
  GEN_TAC THEN REWRITE_TAC[SN_PRESERVE, WN] THEN
  DISCH_THEN(MP_TAC o SPEC (--`\x:'a. ~?y. RTC R x y /\ NORMAL R y`--)) THEN
  BETA_TAC THEN CONV_TAC(DEPTH_CONV NOT_EXISTS_CONV) THEN
  CONV_TAC(ONCE_DEPTH_CONV NOT_FORALL_CONV) THEN
  REWRITE_TAC[] THEN DISCH_THEN MATCH_MP_TAC THEN
  X_GEN_TAC (--`x:'a`--) THEN CONV_TAC(DEPTH_CONV NOT_EXISTS_CONV) THEN
  DISCH_TAC THEN FIRST_ASSUM(MP_TAC o SPEC (--`x:'a`--)) THEN
  REWRITE_TAC[RTC_REFL] THEN
  DISCH_THEN(MP_TAC o REWRITE_RULE[NORMAL]) THEN
  DISCH_THEN(X_CHOOSE_TAC (--`y:'a`--)) THEN
  EXISTS_TAC (--`y:'a`--) THEN ASM_REWRITE_TAC[] THEN
  X_GEN_TAC (--`z:'a`--) THEN STRIP_TAC THEN
  SUBGOAL_THEN (--`RTC R x z /\ NORMAL(R:'a->'a->bool) z`--) MP_TAC THENL
   [CONJ_TAC THENL
     [MATCH_MP_TAC RTC_TRANS_R THEN EXISTS_TAC (--`y:'a`--), ALL_TAC] THEN
    ASM_REWRITE_TAC[],
    REWRITE_TAC[] THEN FIRST_ASSUM MATCH_ACCEPT_TAC]);

(*----------------------------------------------------------------------------*)
(* Reflexive closure preserves Church-Rosser property (pretty trivial)        *)
(*----------------------------------------------------------------------------*)

val RC_CR = prove_thm("RC_CR",
  (--`!R:'a->'a->bool. CR(R) ==> CR(RC R)`--),
  GEN_TAC THEN REWRITE_TAC[CR, RC_CASES] THEN
  CONV_TAC(ONCE_DEPTH_CONV SYM_CONV) THEN
  REPEAT STRIP_TAC THEN ASM_REWRITE_TAC[] THENL
   [FIRST_ASSUM(UNDISCH_TAC o assert is_forall o concl) THEN
    DISCH_THEN(MP_TAC o SPECL [(--`x:'a`--), (--`y1:'a`--), (--`y2:'a`--)]) THEN
    ASM_REWRITE_TAC[] THEN DISCH_THEN(X_CHOOSE_TAC (--`z:'a`--)) THEN
    EXISTS_TAC (--`z:'a`--), EXISTS_TAC (--`y1:'a`--), EXISTS_TAC (--`y2:'a`--),
    EXISTS_TAC (--`x:'a`--)] THEN
  ASM_REWRITE_TAC[]);

(*----------------------------------------------------------------------------*)
(* The strip lemma leads us halfway to the fact that transitive         x     *)
(* closure preserves the Church-Rosser property. It's no harder        / \    *)
(* to prove it for two separate reduction relations. This then        /   y2  *)
(* allows us to prove the desired theorem simply by using the        /    /   *)
(* strip lemma twice with a bit of conjunct-swapping.               y1   /    *)
(*                                                                    \ /     *)
(* The diagram on the right shows the use of the variables.            z      *)
(*----------------------------------------------------------------------------*)

val STRIP_LEMMA = prove_thm("STRIP_LEMMA",
  (--`!R R'. (!x y1 y2.    R x y1 /\ R' x y2 ==> ?z:'a. R' y1 z /\    R y2 z) ==>
         (!x y1 y2. TC R x y1 /\ R' x y2 ==> ?z:'a. R' y1 z /\ TC R y2 z)`--),
  REPEAT GEN_TAC THEN DISCH_TAC THEN
  REWRITE_TAC[TAUT_CONV (--`a /\ b ==> c = a ==> (b ==> c)`--)] THEN
  CONV_TAC(ONCE_DEPTH_CONV FORALL_IMP_CONV) THEN
  RULE_INDUCT_TAC TC_INDUCT THENL
   [GEN_TAC THEN DISCH_THEN(MP_TAC o CONJ(ASSUME (--`(R:'a->'a->bool) x y`--))) THEN
    DISCH_THEN(ANTE_RES_THEN MP_TAC) THEN
    DISCH_THEN(X_CHOOSE_THEN (--`z:'a`--) STRIP_ASSUME_TAC) THEN
    EXISTS_TAC (--`z:'a`--) THEN ASM_REWRITE_TAC[] THEN
    MATCH_MP_TAC TC_INC THEN ASM_REWRITE_TAC[],
    GEN_TAC THEN DISCH_THEN(ANTE_RES_THEN MP_TAC) THEN
    DISCH_THEN(X_CHOOSE_THEN (--`w:'a`--) STRIP_ASSUME_TAC) THEN
    UNDISCH_TAC (--`(R':'a->'a->bool) y w`--) THEN
    DISCH_THEN(ANTE_RES_THEN MP_TAC) THEN
    DISCH_THEN(X_CHOOSE_THEN (--`v:'a`--) STRIP_ASSUME_TAC) THEN
    EXISTS_TAC (--`v:'a`--) THEN ASM_REWRITE_TAC[] THEN
    MATCH_MP_TAC TC_TRANS THEN
    EXISTS_TAC (--`w:'a`--) THEN ASM_REWRITE_TAC[]]);

(*----------------------------------------------------------------------------*)
(* Transitive closure preserves Church-Rosser property.                       *)
(*----------------------------------------------------------------------------*)

val TC_CR = prove_thm("TC_CR",
  (--`!R:'a->'a->bool. CR(R) ==> CR(TC R)`--),
  GEN_TAC THEN REWRITE_TAC[CR] THEN DISCH_TAC THEN
  MATCH_MP_TAC STRIP_LEMMA THEN ONCE_REWRITE_TAC[CONJ_SYM] THEN
  MATCH_MP_TAC STRIP_LEMMA THEN POP_ASSUM MATCH_ACCEPT_TAC);

(*----------------------------------------------------------------------------*)
(* Reflexive transitive closure preserves Church-Rosser property.             *)
(*----------------------------------------------------------------------------*)

val RTC_CR = prove_thm("RTC_CR",
  (--`!R:'a->'a->bool. CR(R) ==> CR(RTC R)`--),
  GEN_TAC THEN DISCH_TAC THEN REWRITE_TAC[RTC] THEN
  MATCH_MP_TAC RC_CR THEN MATCH_MP_TAC TC_CR THEN
  POP_ASSUM ACCEPT_TAC);

(*----------------------------------------------------------------------------*)
(* Equivalent "Church-Rosser" property for the equivalence relation.          *)
(*----------------------------------------------------------------------------*)

val STC_CR = prove_thm("STC_CR",
  (--`!R:'a->'a->bool. CR(RTC R) =
        !x y. RSTC R x y ==> ?z:'a. RTC R x z /\ RTC R y z`--),
  GEN_TAC THEN EQ_TAC THENL
   [DISCH_TAC THEN RULE_INDUCT_TAC RSTC_INDUCT THENL
     [EXISTS_TAC (--`y:'a`--) THEN REWRITE_TAC[RTC_REFL] THEN
      MATCH_MP_TAC RTC_INC THEN ASM_REWRITE_TAC[],
      X_GEN_TAC (--`x:'a`--) THEN EXISTS_TAC (--`x:'a`--) THEN
      REWRITE_TAC[RTC_REFL],
      EXISTS_TAC (--`z:'a`--) THEN ASM_REWRITE_TAC[],
      FIRST_ASSUM(MP_TAC o SPECL [(--`y:'a`--), (--`z':'a`--), (--`z'':'a`--)] o
        REWRITE_RULE[CR]) THEN ASM_REWRITE_TAC[] THEN
      DISCH_THEN(X_CHOOSE_THEN (--`w:'a`--) STRIP_ASSUME_TAC) THEN
      EXISTS_TAC (--`w:'a`--) THEN CONJ_TAC THEN MATCH_MP_TAC RTC_TRANS THENL
       [EXISTS_TAC (--`z':'a`--), EXISTS_TAC (--`z'':'a`--)] THEN
      ASM_REWRITE_TAC[]],
    REWRITE_TAC[CR] THEN DISCH_TAC THEN
    MAP_EVERY X_GEN_TAC [(--`x:'a`--), (--`y1:'a`--), (--`y2:'a`--)] THEN
    STRIP_TAC THEN FIRST_ASSUM MATCH_MP_TAC THEN
    MATCH_MP_TAC RSTC_TRANS THEN EXISTS_TAC (--`x:'a`--) THEN
    CONJ_TAC THENL [MATCH_MP_TAC RSTC_SYM, ALL_TAC] THEN
    MATCH_MP_TAC RSTC_INC_RTC THEN ASM_REWRITE_TAC[]]);

(*----------------------------------------------------------------------------*)
(* Under normalization, Church-Rosser is equivalent to uniqueness of NF       *)
(*----------------------------------------------------------------------------*)

val NORM_CR = prove_thm("NORM_CR",
  (--`!R:'a->'a->bool. WN(R) ==>
     (CR(RTC R) = (!x y1 y2. RTC R x y1 /\ NORMAL(R) y1 /\
                             RTC R x y2 /\ NORMAL(R) y2 ==> (y1 = y2)))`--),
  GEN_TAC THEN REWRITE_TAC[WN] THEN DISCH_TAC THEN EQ_TAC THEN
  REWRITE_TAC[CR] THEN DISCH_TAC THEN
  MAP_EVERY X_GEN_TAC [(--`x:'a`--), (--`y1:'a`--), (--`y2:'a`--)] THEN STRIP_TAC THENL
   [SUBGOAL_THEN (--`?z. RTC (R:'a->'a->bool) y1 z /\ RTC R y2 z`--) MP_TAC THENL
     [FIRST_ASSUM MATCH_MP_TAC THEN EXISTS_TAC (--`x:'a`--) THEN
      CONJ_TAC THEN FIRST_ASSUM MATCH_ACCEPT_TAC,
      DISCH_THEN(X_CHOOSE_THEN (--`z:'a`--) MP_TAC) THEN
      REWRITE_TAC[RTC, RC_CASES] THEN
      RULE_ASSUM_TAC(ONCE_REWRITE_RULE[GSYM NORMAL_TC]) THEN
      RULE_ASSUM_TAC(REWRITE_RULE[NORMAL]) THEN
      RULE_ASSUM_TAC(CONV_RULE(ONCE_DEPTH_CONV NOT_EXISTS_CONV)) THEN
      ASM_REWRITE_TAC[] THEN DISCH_THEN(CONJUNCTS_THEN SUBST1_TAC) THEN
      REFL_TAC],
    FIRST_ASSUM(X_CHOOSE_THEN (--`z1:'a`--) STRIP_ASSUME_TAC o SPEC (--`y1:'a`--)) THEN
    FIRST_ASSUM(X_CHOOSE_THEN (--`z2:'a`--) STRIP_ASSUME_TAC o SPEC (--`y2:'a`--)) THEN
    EXISTS_TAC (--`z1:'a`--) THEN
    SUBGOAL_THEN (--`z1:'a = z2`--) (fn th =>  ASM_REWRITE_TAC[th]) THEN
    FIRST_ASSUM MATCH_MP_TAC THEN EXISTS_TAC (--`x:'a`--) THEN
    ASM_REWRITE_TAC[] THEN CONJ_TAC THEN
    MATCH_MP_TAC RTC_TRANS THENL
     [EXISTS_TAC (--`y1:'a`--), EXISTS_TAC (--`y2:'a`--)] THEN
    ASM_REWRITE_TAC[]]);

(*----------------------------------------------------------------------------*)
(* Normalizing and Church-Rosser iff every term has a unique normal form      *)
(*----------------------------------------------------------------------------*)

val CR_NORM = prove_thm("CR_NORM",
  (--`!R:'a->'a->bool. WN(R) /\ CR(RTC R) = !x. ?!y. RTC R x y /\ NORMAL(R) y`--),
  GEN_TAC THEN CONV_TAC(ONCE_DEPTH_CONV EXISTS_UNIQUE_CONV) THEN
  CONV_TAC(ONCE_DEPTH_CONV FORALL_AND_CONV) THEN
  REWRITE_TAC[GSYM WN, TAUT_CONV (--`(a /\ b = a /\ c) = a ==> (b = c)`--)] THEN
  DISCH_THEN(SUBST1_TAC o MATCH_MP NORM_CR) THEN
  REWRITE_TAC[CONJ_ASSOC]);

(*----------------------------------------------------------------------------*)
(* Newman's lemma: weak Church-Rosser plus                   x                *)
(* strong normalization implies full Church-                / \               *)
(* Rosser. By the above (and SN ==> WN) it                 z1 z2              *)
(* is sufficient to show normal forms are                 / | | \             *)
(* unique. We use the Noetherian induction               /  \ /  \            *)
(* form of SN, so we need only prove that if            /    z    \           *)
(* some term has multiple normal forms, so             /     |     \          *)
(* does a "successor". See the diagram on the         /      |      \         *)
(* right for the use of variables.                   y1      w       y2       *)
(*----------------------------------------------------------------------------*)

val NEWMAN_LEMMA = prove_thm("NEWMAN_LEMMA",
  (--`!R:'a->'a->bool. SN(R) /\ WCR(R) ==> CR(RTC R)`--),
  REPEAT STRIP_TAC THEN FIRST_ASSUM(ASSUME_TAC o MATCH_MP SN_WN) THEN
  FIRST_ASSUM(fn th =>  REWRITE_TAC[MATCH_MP NORM_CR th]) THEN
  GEN_REWRITE_TAC I [] [TAUT_CONV (--`x = ~~x`--)] THEN
  CONV_TAC(TOP_DEPTH_CONV NOT_FORALL_CONV) THEN
  REWRITE_TAC[NOT_IMP, GSYM CONJ_ASSOC] THEN
  MP_TAC(SPEC (--`R:'a->'a->bool`--) SN_PRESERVE) THEN
  ASM_REWRITE_TAC[] THEN CONV_TAC(ONCE_DEPTH_CONV NOT_FORALL_CONV) THEN
  REWRITE_TAC[] THEN CONV_TAC(ONCE_DEPTH_CONV ETA_CONV) THEN
  DISCH_THEN MATCH_MP_TAC THEN BETA_TAC THEN
  GEN_TAC THEN REWRITE_TAC[NORMAL] THEN
  CONV_TAC(ONCE_DEPTH_CONV NOT_EXISTS_CONV) THEN STRIP_TAC THEN
  MP_TAC(ASSUME (--`RTC (R:'a->'a->bool) x y1`--)) THEN
  DISCH_THEN(MP_TAC o ONCE_REWRITE_RULE[RTC_CASES_R]) THEN
  DISCH_THEN(DISJ_CASES_THEN2 (SUBST_ALL_TAC o SYM) MP_TAC) THENL
   [UNDISCH_TAC (--`RTC(R:'a->'a->bool) x y2`--) THEN
    ONCE_REWRITE_TAC[RTC_CASES_R] THEN
    FIRST_ASSUM(fn th =>  REWRITE_TAC[CONV_RULE(RAND_CONV SYM_CONV) th]) THEN
    ASM_REWRITE_TAC[],
    DISCH_THEN(X_CHOOSE_THEN (--`z1:'a`--) STRIP_ASSUME_TAC)] THEN
  MP_TAC(ASSUME (--`RTC (R:'a->'a->bool) x y2`--)) THEN
  DISCH_THEN(MP_TAC o ONCE_REWRITE_RULE[RTC_CASES_R]) THEN
  DISCH_THEN(DISJ_CASES_THEN2 (SUBST_ALL_TAC o SYM) MP_TAC) THENL
   [UNDISCH_TAC (--`RTC(R:'a->'a->bool) x y1`--) THEN
    ONCE_REWRITE_TAC[RTC_CASES_R] THEN
    FIRST_ASSUM(fn th =>  REWRITE_TAC[CONV_RULE(RAND_CONV SYM_CONV) th]) THEN
    ASM_REWRITE_TAC[],
    DISCH_THEN(X_CHOOSE_THEN (--`z2:'a`--) STRIP_ASSUME_TAC)] THEN
  UNDISCH_TAC (--`WCR(R:'a->'a->bool)`--) THEN REWRITE_TAC[WCR] THEN
  DISCH_THEN(MP_TAC o SPECL [(--`x:'a`--), (--`z1:'a`--), (--`z2:'a`--)]) THEN
  ASM_REWRITE_TAC[] THEN DISCH_THEN(X_CHOOSE_THEN (--`z:'a`--) STRIP_ASSUME_TAC) THEN
  UNDISCH_TAC (--`WN(R:'a->'a->bool)`--) THEN REWRITE_TAC[WN] THEN
  REWRITE_TAC[NORMAL] THEN CONV_TAC(ONCE_DEPTH_CONV NOT_EXISTS_CONV) THEN
  DISCH_THEN(X_CHOOSE_THEN (--`w:'a`--) STRIP_ASSUME_TAC o SPEC (--`z:'a`--)) THEN
  ASM_CASES_TAC (--`y1:'a = w`--) THENL
   [ALL_TAC,
    EXISTS_TAC (--`z1:'a`--) THEN ASM_REWRITE_TAC[] THEN
    MAP_EVERY EXISTS_TAC [(--`y1:'a`--), (--`w:'a`--)] THEN ASM_REWRITE_TAC[] THEN
    MATCH_MP_TAC RTC_TRANS THEN EXISTS_TAC (--`z:'a`--) THEN
    ASM_REWRITE_TAC[]] THEN
  ASM_CASES_TAC (--`y2:'a = w`--) THENL
   [ALL_TAC,
    EXISTS_TAC (--`z2:'a`--) THEN ASM_REWRITE_TAC[] THEN
    MAP_EVERY EXISTS_TAC [(--`y2:'a`--), (--`w:'a`--)] THEN ASM_REWRITE_TAC[] THEN
    MATCH_MP_TAC RTC_TRANS THEN EXISTS_TAC (--`z:'a`--) THEN
    ASM_REWRITE_TAC[]] THEN
  UNDISCH_TAC (--`~(y1:'a = y2)`--) THEN ASM_REWRITE_TAC[]);

close_theory();

export_theory();
