/***************************************************************************
                          msnsocketbase.cpp  -  description
                             -------------------
    begin                : Thu Apr 1 2008
    copyright            : (C) 2008 by Valerio Pilo
    email                : valerio@kmess.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "msnsocketbase.h"

#include <QAuthenticator>
#include <QNetworkProxy>

#include <KLocale>
#include <KPasswordDialog>
#include <kdeversion.h>


/**
 * @brief The constructor
 */
MsnSocketBase::MsnSocketBase( ServerType serverType )
: connected_( false )
{
  // Set the type of server the derived socket will connect to
  serverType_ = serverType;
}



/**
 * @brief The destructor
 *
 * Does nothing.
 */
MsnSocketBase::~MsnSocketBase()
{
}



/**
 * @brief Return which payload commands are accepted by this connection
 *
 * Returns the list of payload messages which this connection needs to accept.
 *
 * @return List of 3-letter commands which are recognized as carrying payload.
 */
QStringList MsnSocketBase::getAcceptedPayloadCommands()
{
  return acceptedPayloadCommands_;
}



/**
 * @brief Return the type of server connection managed by the socket
 */
MsnSocketBase::ServerType MsnSocketBase::getServerType() const
{
  return serverType_;
}



/**
 * @brief Whether or not the connection is active
 *
 * @returns True when connected, false otherwise.
 */
bool MsnSocketBase::isConnected() const
{
  return ( connected_ );
}



/**
 * @brief Test whether the given command is an error.
 *
 * @param command Command to verify
 * @returns True when it's an error command, false otherwise.
 */
bool MsnSocketBase::isErrorCommand( const QString &command ) const
{
  bool isNumeric = false;
  return ( command.toInt( &isNumeric ) != 0 ) && ( isNumeric == true );
}



/**
 * @brief Test whether the given command is a payload command.
 *
 * @param command Command to verify
 * @returns True when it's a payload command, false otherwise.
 */
bool MsnSocketBase::isPayloadCommand( const QString &command ) const
{
  return acceptedPayloadCommands_.contains( command );
}



// Ask the user to authenticate on a proxy
void MsnSocketBase::proxyAuthenticate( const QNetworkProxy &proxy, QAuthenticator *authenticator )
{
#ifdef KMESSDEBUG_SHARED_PROXY_AUTH
  kmDebug() << "Authentication to proxy" << proxy.hostName() << "required.";
  kmDebug() << "Realm:" << authenticator->realm() << "Username:" << authenticator->user();
#else
  Q_UNUSED( proxy );
#endif

  KPasswordDialog::KPasswordDialogFlags flags =
                                                KPasswordDialog::ShowUsernameLine
#if KDE_IS_VERSION( 4, 1, 0 )
                                              | KPasswordDialog::ShowDomainLine
                                              | KPasswordDialog::DomainReadOnly
                                              | KPasswordDialog::ShowAnonymousLoginCheckBox
#endif
                                              | KPasswordDialog::ShowKeepPassword;

  KPasswordDialog loginDialog( 0, flags );
  loginDialog.setPrompt( i18nc( "Dialog box message",
                                "Enter your username and password to access the network proxy." ) );

  // The user canceled
  if( ! loginDialog.exec() )
  {
      return;
  }

  authenticator->setUser( loginDialog.username() );
  authenticator->setPassword( loginDialog.password() );
}



/**
 * @brief Specify which accepted commands carry payload data for this connection.
 *
 * Within different kinds of connection, server messages can be different or
 * have different meanings; this method ensures that every kind of connection
 * is able to recognize its own payload commands only.
 *
 * @param commandList List of 3-letter commands which will be recognized as carrying payload.
 */
void MsnSocketBase::setAcceptedPayloadCommands( QStringList commandList )
{
  acceptedPayloadCommands_ = commandList;
}



#include "msnsocketbase.moc"
